from django.db import models
from django.conf import settings
from django.utils.translation import gettext_lazy as _
import uuid

class Conversation(models.Model):
    """
    Représente une session de chat avec un visiteur.
    Identifiée par un UUID pour éviter d'exposer des IDs séquentiels.
    """
    uuid = models.UUIDField(default=uuid.uuid4, editable=False, unique=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    
    # Métriques simples
    message_count = models.PositiveIntegerField(default=0)

    class Meta:
        ordering = ['-updated_at']
        verbose_name = _("Conversation IA")
        verbose_name_plural = _("Conversations IA")

    def __str__(self):
        return f"Conversation {str(self.uuid)[:8]}... ({self.created_at.strftime('%d/%m %H:%M')})"


class ChatMessage(models.Model):
    """
    Chaque message échangé (User ou Assistant).
    """
    ROLE_CHOICES = (
        ('user', 'Utilisateur'),
        ('assistant', 'IA (Assistant)'),
        ('system', 'Système (Interne)'),
    )

    conversation = models.ForeignKey(
        Conversation, 
        related_name='messages', 
        on_delete=models.CASCADE
    )
    role = models.CharField(max_length=20, choices=ROLE_CHOICES)
    content = models.TextField()
    created_at = models.DateTimeField(auto_now_add=True)
    
    # Pour le debug : savoir si le message a utilisé le contexte RAG
    used_context = models.BooleanField(default=False)

    class Meta:
        ordering = ['created_at']
        verbose_name = _("Message")

    def __str__(self):
        return f"{self.role}: {self.content[:50]}..."