from django.db import models
from django.utils.translation import gettext_lazy as _
from apps.pages.models import Page

class Menu(models.Model):
    """
    Conteneur de liens (ex: 'main_navbar', 'footer_links').
    """
    title = models.CharField(max_length=100, verbose_name=_("Nom du menu"))
    slug = models.SlugField(unique=True, help_text=_("Identifiant unique pour le code (ex: main_navbar)"))

    class Meta:
        verbose_name = _("Menu")
        verbose_name_plural = _("Menus")

    def __str__(self):
        return self.title


class MenuItem(models.Model):
    """
    Un lien dans un menu. Peut être parent d'autres liens.
    """
    menu = models.ForeignKey(
        Menu, 
        related_name='items', 
        on_delete=models.CASCADE
    )
    parent = models.ForeignKey(
        'self', 
        null=True, 
        blank=True, 
        related_name='children', 
        on_delete=models.CASCADE,
        verbose_name=_("Parent (Sous-menu)")
    )
    
    title = models.CharField(max_length=100, verbose_name=_("Libellé"))
    
    # Stratégie hybride : Soit une page interne, soit une URL libre
    page = models.ForeignKey(
        Page, 
        null=True, 
        blank=True, 
        on_delete=models.SET_NULL,
        verbose_name=_("Page Interne"),
        help_text=_("Si sélectionné, l'URL sera générée automatiquement.")
    )
    external_url = models.CharField(
        max_length=255, 
        blank=True, 
        verbose_name=_("URL Externe / Manuelle"),
        help_text=_("Utilisé seulement si aucune page interne n'est sélectionnée.")
    )
    
    order = models.PositiveIntegerField(default=0, verbose_name=_("Ordre"))

    class Meta:
        ordering = ['order']
        verbose_name = _("Élément de menu")
        verbose_name_plural = _("Éléments de menu")

    def __str__(self):
        return self.title

    @property
    def url(self):
        """Retourne l'URL calculée (Page ou Externe)"""
        if self.page and self.page.slug:
            # On suppose que le frontend gère les routes par slug
            if self.page.slug == 'accueil':
                return '/'
            return f"/{self.page.slug}"
        return self.external_url or "#"