from django.db import models
from django.utils.translation import gettext_lazy as _
from django.utils.text import slugify

class Page(models.Model):
    """
    Représente une page du site (Home, About, etc.).
    Sert de conteneur pour les Sections.
    """
    title = models.CharField(max_length=200, verbose_name=_("Titre de la page"))
    slug = models.SlugField(max_length=200, unique=True, verbose_name=_("URL Slug"))
    
    # SEO & Métadonnées
    meta_description = models.CharField(
        max_length=160, 
        blank=True, 
        verbose_name=_("Méta Description (SEO)"),
        help_text=_("Description courte pour Google (160 caractères max).")
    )
    
    # État de publication
    is_published = models.BooleanField(default=False, verbose_name=_("Publié"))
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        verbose_name = _("Page")
        verbose_name_plural = _("Pages")
        ordering = ['title']

    def __str__(self):
        return self.title

    def save(self, *args, **kwargs):
        # Génération automatique du slug si vide (sauf pour la home souvent fixée manuellement)
        if not self.slug:
            self.slug = slugify(self.title)
        super().save(*args, **kwargs)