from rest_framework import serializers
from .models import Page
from apps.sections.models import Section
from apps.sections.serializers import get_serializer_class_for_type

class SectionDetailSerializer(serializers.ModelSerializer):
    """
    Serializer hybride :
    - Rend les champs DB de la section (id, type)
    - Rend le champ 'data' validé par le serializer spécifique du type
    """
    content = serializers.SerializerMethodField()

    class Meta:
        model = Section
        fields = ['id', 'type', 'order', 'content']

    def get_content(self, obj):
        """
        Transforme le JSON brut en données validées selon le type.
        Ex: Si type='hero', utilise HeroSerializer.
        """
        SerializerClass = get_serializer_class_for_type(obj.type)
        # On passe le contenu JSON (obj.data) au serializer spécifique
        serializer = SerializerClass(data=obj.data)
        
        # Note: Les données sont déjà validées en DB via clean(), 
        # mais on re-vérifie pour formater proprement (ex: URL absolues pour images)
        if serializer.is_valid():
            return serializer.data
        return obj.data  # Fallback en cas d'erreur (ne devrait pas arriver)

class PageDetailSerializer(serializers.ModelSerializer):
    """
    Serializer complet pour une page (utilisé par Next.js).
    Inclut la liste des sections actives.
    """
    sections = serializers.SerializerMethodField()

    class Meta:
        model = Page
        fields = ['id', 'title', 'slug', 'meta_description', 'sections']

    def get_sections(self, obj):
        # On récupère uniquement les sections actives et ordonnées
        qs = obj.sections.filter(is_active=True).order_by('order')
        return SectionDetailSerializer(qs, many=True, context=self.context).data

class PageListSerializer(serializers.ModelSerializer):
    """Serializer léger pour les listes (menus, sitemap)"""
    class Meta:
        model = Page
        fields = ['id', 'title', 'slug']