from django.db import models
from django.core.exceptions import ValidationError
from django.utils.translation import gettext_lazy as _

from .definitions import SectionType
from .serializers import get_serializer_class_for_type

class Section(models.Model):
    """
    Une section est un bloc de contenu rattaché à une page.
    Elle contient des données structurées en JSON (le champ 'data').
    """
    # Relation Lazy vers Page (évite l'import circulaire)
    page = models.ForeignKey(
        'pages.Page', 
        on_delete=models.CASCADE, 
        related_name='sections'
    )
    
    type = models.CharField(
        max_length=50, 
        choices=SectionType.choices,
        db_index=True
    )
    
    # Nom interne pour l'admin (ex: "Hero Campagne Noël")
    internal_name = models.CharField(
        max_length=100, 
        blank=True, 
        help_text=_("Nom visible uniquement dans l'admin pour s'organiser")
    )
    
    # Le cœur du CMS : Stockage des données
    data = models.JSONField(default=dict, blank=True)
    
    # Gestion de l'ordre d'affichage
    order = models.PositiveIntegerField(
        default=0, 
        blank=False, 
        null=False,
        db_index=True
    )
    
    is_active = models.BooleanField(default=True)
    
    class Meta:
        ordering = ['order']
        verbose_name = _("Section")
        verbose_name_plural = _("Sections")

    def __str__(self):
        return f"{self.get_type_display()} - {self.internal_name or 'Sans nom'}"

    def clean(self):
        """
        Validation métier avant sauvegarde.
        Vérifie que le JSON 'data' respecte le schéma du 'type' choisi.
        """
        super().clean()
        
        # Récupération du validateur (Serializer) correspondant au type
        SerializerClass = get_serializer_class_for_type(self.type)
        serializer = SerializerClass(data=self.data)
        
        if not serializer.is_valid():
            # On transforme les erreurs API en erreurs Admin lisibles
            raise ValidationError({'data': serializer.errors})
            
        # (Optionnel) On pourrait nettoyer les données ici
        # self.data = serializer.validated_data

    def save(self, *args, **kwargs):
        self.full_clean() # Force la validation clean() à chaque sauvegarde
        super().save(*args, **kwargs)