from django.db import models
from django.core.exceptions import ValidationError
from django.utils.translation import gettext_lazy as _

class GlobalSettings(models.Model):
    """
    Singleton pour la configuration globale du site.
    """
    # --- 1. IDENTITÉ VISUELLE ---
    site_name = models.CharField(max_length=100, default="Mon Site Headless")
    site_description = models.TextField(
        max_length=300, 
        blank=True, 
        help_text=_("Description par défaut pour le SEO (si la page n'en a pas).")
    )
    logo_light = models.ImageField(
        upload_to='settings/logos/', 
        blank=True, 
        null=True, 
        verbose_name=_("Logo (Fond Clair)")
    )
    logo_dark = models.ImageField(
        upload_to='settings/logos/', 
        blank=True, 
        null=True, 
        verbose_name=_("Logo (Fond Sombre)")
    )
    favicon = models.ImageField(upload_to='settings/favicons/', blank=True, null=True)

    # --- 2. COORDONNÉES (Source de vérité) ---
    contact_email = models.EmailField(blank=True, verbose_name=_("Email public"))
    contact_phone = models.CharField(max_length=20, blank=True, verbose_name=_("Téléphone public"))
    contact_address = models.TextField(blank=True, verbose_name=_("Adresse postale"))

    # --- 3. RÉSEAUX SOCIAUX (JSON) ---
    social_links = models.JSONField(
        default=dict, 
        blank=True, 
        help_text=_("Format: {'facebook': 'url', 'linkedin': 'url'}")
    )

    # --- 4. TECHNIQUE ---
    google_analytics_id = models.CharField(max_length=50, blank=True, verbose_name="GA4 ID")
    is_maintenance_mode = models.BooleanField(default=False, verbose_name="Mode Maintenance")

    class Meta:
        verbose_name = _("Configuration Globale")
        verbose_name_plural = _("Configuration Globale")

    def __str__(self):
        return "Paramètres du Site"

    def save(self, *args, **kwargs):
        """Empêche la création de multiples configurations"""
        if not self.pk and GlobalSettings.objects.exists():
            raise ValidationError("Il ne peut y avoir qu'une seule configuration globale.")
        return super(GlobalSettings, self).save(*args, **kwargs)